/*
 *  include/ub/beancounter.h
 *
 *  Copyright (C) 1999-2005  SWsoft
 *  All rights reserved.
 *
 *  Andrey Savochkin	saw@sw-soft.com
 *
 */

#ifndef _LINUX_BEANCOUNTER_H
#define _LINUX_BEANCOUNTER_H

#include <linux/config.h>

/*
 * Generic ratelimiting stuff.
 */

struct ub_rate_info {
	int burst;
	int interval; /* jiffy_t per event */
	int bucket; /* kind of leaky bucket */
	unsigned long last; /* last event */
};

/* Return true if rate limit permits. */
int ub_ratelimit(struct ub_rate_info *);


/*
 * This magic is used to distinuish user beancounter and pages beancounter
 * in struct page. page_ub and page_bc are placed in union and MAGIC
 * ensures us that we don't use pbc as ubc in ub_page_uncharge().
 */
#define UB_MAGIC		0x62756275

/*
 *	Resource list.
 */

#define UB_KMEMSIZE	0	/* Unswappable kernel memory size including
				 * struct task, page directories, etc.
				 */
#define UB_LOCKEDPAGES	1	/* Mlock()ed pages. */
#define UB_PRIVVMPAGES	2	/* Total number of pages, counting potentially
				 * private pages as private and used.
				 */
#define UB_SHMPAGES	3	/* IPC SHM segment size. */
#define UB_ZSHMPAGES	4	/* Anonymous shared memory. */
#define UB_NUMPROC	5	/* Number of processes. */
#define UB_PHYSPAGES	6	/* All resident pages, for swapout guarantee. */
#define UB_VMGUARPAGES	7	/* Guarantee for memory allocation,
				 * checked against PRIVVMPAGES.
				 */
#define UB_OOMGUARPAGES	8	/* Guarantees against OOM kill.
				 * Only limit is used, no accounting.
				 */
#define UB_NUMTCPSOCK	9	/* Number of TCP sockets. */
#define UB_NUMFLOCK	10	/* Number of file locks. */
#define UB_NUMPTY	11	/* Number of PTYs. */
#define UB_NUMSIGINFO	12	/* Number of siginfos. */
#define UB_TCPSNDBUF	13	/* Total size of tcp send buffers. */
#define UB_TCPRCVBUF	14	/* Total size of tcp receive buffers. */
#define UB_OTHERSOCKBUF	15	/* Total size of other socket
				 * send buffers (all buffers for PF_UNIX).
				 */
#define UB_DGRAMRCVBUF	16	/* Total size of other socket
				 * receive buffers.
				 */
#define UB_NUMOTHERSOCK	17	/* Number of other sockets. */
#define UB_DCACHESIZE	18	/* Size of busy dentry/inode cache. */
#define UB_NUMFILE	19	/* Number of open files. */

#define UB_RESOURCES	24

#define UB_UNUSEDPRIVVM	(UB_RESOURCES + 0)
#define UB_TMPFSPAGES	(UB_RESOURCES + 1)
#define UB_SWAPPAGES	(UB_RESOURCES + 2)
#define UB_HELDPAGES	(UB_RESOURCES + 3)

struct ubparm {
	/* 
	 * A barrier over which resource allocations are failed gracefully.
	 * If the amount of consumed memory is over the barrier further sbrk()
	 * or mmap() calls fail, the existing processes are not killed. 
	 */
	unsigned long	barrier;
	/* hard resource limit */
	unsigned long	limit;
	/* consumed resources */
	unsigned long	held;
	/* maximum amount of consumed resources through the last period */
	unsigned long	maxheld;
	/* minimum amount of consumed resources through the last period */
	unsigned long	minheld;
	/* count of failed charges */
	unsigned long	failcnt;
};

/*
 * Kernel internal part.
 */

#ifdef __KERNEL__

#include <ub/ub_debug.h>
#include <linux/interrupt.h>
#include <asm/atomic.h>
#include <linux/spinlock.h>
#include <linux/cache.h>
#include <linux/threads.h>

/*
 * UB_MAXVALUE is essentially LONG_MAX declared in a cross-compiling safe form.
 */
#define UB_MAXVALUE	( (1UL << (sizeof(unsigned long)*8-1)) - 1)


/*
 *	Resource management structures
 * Serialization issues:
 *   beancounter list management is protected via ub_hash_lock
 *   task pointers are set only for current task and only once
 *   refcount is managed atomically
 *   value and limit comparison and change are protected by per-ub spinlock
 */

struct page_beancounter;
struct task_beancounter;
struct sock_beancounter;

struct page_private {
	unsigned long		ubp_unused_privvmpages;
	unsigned long		ubp_tmpfs_respages;
	unsigned long		ubp_swap_pages;
	unsigned long long	ubp_held_pages;
};

struct sock_private {
	unsigned long		ubp_rmem_thres;
	unsigned long		ubp_wmem_pressure;
	unsigned long		ubp_maxadvmss;
	unsigned long		ubp_rmem_pressure;
#define UB_RMEM_EXPAND          0
#define UB_RMEM_KEEP            1
#define UB_RMEM_SHRINK          2
	struct list_head	ubp_other_socks;
	struct list_head	ubp_tcp_socks;
	atomic_t		ubp_orphan_count;
};

struct ub_perfstat {
	unsigned long unmap;
	unsigned long swapin;
} ____cacheline_aligned_in_smp;

struct user_beancounter
{
	unsigned long		ub_magic;
	atomic_t		ub_refcount;
	struct			user_beancounter *ub_next;
	spinlock_t		ub_lock;
	uid_t			ub_uid;

	struct ub_rate_info	ub_limit_rl;
	int			ub_oom_noproc;

	struct page_private	ppriv;
#define ub_unused_privvmpages	ppriv.ubp_unused_privvmpages
#define ub_tmpfs_respages	ppriv.ubp_tmpfs_respages
#define ub_swap_pages		ppriv.ubp_swap_pages
#define ub_held_pages		ppriv.ubp_held_pages
	struct sock_private	spriv;
#define ub_rmem_thres		spriv.ubp_rmem_thres
#define ub_maxadvmss		spriv.ubp_maxadvmss
#define ub_rmem_pressure	spriv.ubp_rmem_pressure
#define ub_wmem_pressure	spriv.ubp_wmem_pressure
#define ub_tcp_sk_list		spriv.ubp_tcp_socks
#define ub_other_sk_list	spriv.ubp_other_socks
#define ub_orphan_count		spriv.ubp_orphan_count

	struct user_beancounter *parent;
	void *private_data;

	/* resources statistic and settings */
	struct ubparm		ub_parms[UB_RESOURCES];
	/* resources statistic for last interval */
	struct ubparm		ub_store[UB_RESOURCES];

	struct ub_perfstat	ub_perfstat[NR_CPUS];

#ifdef CONFIG_UBC_DEBUG_KMEM
	struct list_head	ub_cclist;
	unsigned int		ub_pages_charged[NR_CPUS];
	unsigned int		ub_vmalloc_charged[NR_CPUS];
#endif
};

enum severity { UB_HARD, UB_SOFT, UB_FORCE };

static inline int ub_barrier_hit(struct user_beancounter *ub, int resource)
{
	return ub->ub_parms[resource].held > ub->ub_parms[resource].barrier;
}

static inline int ub_hfbarrier_hit(struct user_beancounter *ub, int resource)
{
	return (ub->ub_parms[resource].held > 
		((ub->ub_parms[resource].barrier) >> 1));
}

#ifndef CONFIG_USER_RESOURCE

extern inline struct user_beancounter *get_beancounter_byuid
		(uid_t uid, int create) { return NULL; }
extern inline struct user_beancounter *get_beancounter
		(struct user_beancounter *ub) { return NULL; }
extern inline void put_beancounter(struct user_beancounter *ub) {;}

static inline void page_ubc_init(void) { };
static inline void beancounter_init(unsigned long mempages) { };
static inline void ub0_init(void) { };

#else /* CONFIG_USER_RESOURCE */

/*
 *  Charge/uncharge operations
 */

extern int __charge_beancounter_locked(struct user_beancounter *ub,
		int resource, unsigned long val, enum severity strict);

extern void __uncharge_beancounter_locked(struct user_beancounter *ub,
		int resource, unsigned long val);

extern void __put_beancounter(struct user_beancounter *ub);

extern void uncharge_warn(struct user_beancounter *ub, int resource,
		unsigned long val, unsigned long held);

extern const char *ub_rnames[];
/*
 *	Put a beancounter reference
 */

static inline void put_beancounter(struct user_beancounter *ub)
{
	if (unlikely(ub == NULL))
		return;

	__put_beancounter(ub);
}

/*
 *	Create a new beancounter reference
 */
extern struct user_beancounter *get_beancounter_byuid(uid_t uid, int create);

static inline 
struct user_beancounter *get_beancounter(struct user_beancounter *ub)
{
	if (unlikely(ub == NULL))
		return NULL;

	atomic_inc(&ub->ub_refcount);
	return ub;
}

extern struct user_beancounter *get_subbeancounter_byid(
		struct user_beancounter *,
		int id, int create);

extern void beancounter_init(unsigned long);
extern void page_ubc_init(void);
extern struct user_beancounter ub0;
extern void ub0_init(void);
#define get_ub0()	(&ub0)

extern void print_ub_uid(struct user_beancounter *ub, char *buf, int size);

/*
 *	Resource charging
 * Change user's account and compare against limits
 */

static inline void ub_adjust_maxheld(struct user_beancounter *ub, int resource)
{
	if (ub->ub_parms[resource].maxheld < ub->ub_parms[resource].held)
		ub->ub_parms[resource].maxheld = ub->ub_parms[resource].held;
	if (ub->ub_parms[resource].minheld > ub->ub_parms[resource].held)
		ub->ub_parms[resource].minheld = ub->ub_parms[resource].held;
}

#endif /* CONFIG_USER_RESOURCE */

#include <ub/ub_decl.h>
UB_DECLARE_FUNC(int, charge_beancounter(struct user_beancounter *ub,
			int resource, unsigned long val, enum severity strict));
UB_DECLARE_VOID_FUNC(uncharge_beancounter(struct user_beancounter *ub,
			int resource, unsigned long val));

UB_DECLARE_VOID_FUNC(charge_beancounter_notop(struct user_beancounter *ub,
			int resource, unsigned long val));
UB_DECLARE_VOID_FUNC(uncharge_beancounter_notop(struct user_beancounter *ub,
			int resource, unsigned long val));

#ifndef CONFIG_USER_RESOURCE_PROC
static inline void beancounter_proc_init(void) { };
#else
extern void beancounter_proc_init(void);
#endif
#endif /* __KERNEL__ */
#endif /* _LINUX_BEANCOUNTER_H */
